import { useState, useEffect } from 'react'
import { motion } from 'framer-motion'
import { Briefcase, User, Star, TrendingUp, Award, Target, BarChart3, CheckCircle2, ArrowRight, Calendar, Sparkles, Zap } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MySessions.css'

const MySessions = () => {
  const [sessions, setSessions] = useState([])

  useEffect(() => {
    const saved = localStorage.getItem('careerSessions')
    if (saved) {
      try {
        setSessions(JSON.parse(saved))
      } catch (e) {
        setSessions([])
      }
    } else {
      const defaultSessions = [
        {
          id: 1,
          consultant: 'Елена Смирнова',
          topic: 'Подготовка резюме',
          result: 'Резюме обновлено, получено 3 приглашения на собеседование',
          review: 'Отличная консультация, помогла структурировать опыт',
          rating: 5,
          date: '2025-03-25'
        },
        {
          id: 2,
          consultant: 'Дмитрий Петров',
          topic: 'Собеседование',
          result: 'Успешно прошел собеседование, получил оффер',
          review: 'Профессиональная подготовка, все советы пригодились',
          rating: 5,
          date: '2025-03-20'
        },
        {
          id: 3,
          consultant: 'Анна Козлова',
          topic: 'Смена профессии',
          result: 'Определен план перехода в новую сферу',
          review: 'Помогла понять, куда двигаться дальше',
          rating: 4,
          date: '2025-03-15'
        }
      ]
      setSessions(defaultSessions)
      localStorage.setItem('careerSessions', JSON.stringify(defaultSessions))
    }
  }, [])

  const totalSessions = sessions.length
  const averageRating = sessions.length > 0
    ? (sessions.reduce((sum, s) => sum + s.rating, 0) / sessions.length).toFixed(1)
    : 0
  const uniqueConsultants = new Set(sessions.map(s => s.consultant)).size
  const successfulResults = sessions.filter(s => s.result.toLowerCase().includes('успешно') || s.result.toLowerCase().includes('получен')).length

  return (
    <div className="my-sessions">
      <div className="container">
        <motion.section 
          className="hero-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6 }}
        >
          <h1 className="hero-title">Мои сессии</h1>
          <p className="hero-description">
            Отслеживайте свой карьерный прогресс
          </p>
        </motion.section>

        <motion.section 
          className="stats-minimal-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.2 }}
        >
          <div className="stats-minimal-grid">
            <motion.div 
              className="stat-minimal-item"
              whileHover={{ x: 4, scale: 1.02, transition: { duration: 0.2 } }}
            >
              <div className="stat-minimal-icon">
                <Briefcase size={32} strokeWidth={2} color="#FF6B6B" />
              </div>
              <div className="stat-minimal-content">
                <div className="stat-minimal-value">{totalSessions}</div>
                <div className="stat-minimal-label">Сессий</div>
              </div>
            </motion.div>
            <motion.div 
              className="stat-minimal-item"
              whileHover={{ x: 4, scale: 1.02, transition: { duration: 0.2 } }}
            >
              <div className="stat-minimal-icon">
                <Star size={32} strokeWidth={2} color="#FFA07A" />
              </div>
              <div className="stat-minimal-content">
                <div className="stat-minimal-value">{averageRating}</div>
                <div className="stat-minimal-label">Оценка</div>
              </div>
            </motion.div>
            <motion.div 
              className="stat-minimal-item"
              whileHover={{ x: 4, scale: 1.02, transition: { duration: 0.2 } }}
            >
              <div className="stat-minimal-icon">
                <User size={32} strokeWidth={2} color="#FFB6C1" />
              </div>
              <div className="stat-minimal-content">
                <div className="stat-minimal-value">{uniqueConsultants}</div>
                <div className="stat-minimal-label">Консультантов</div>
              </div>
            </motion.div>
            <motion.div 
              className="stat-minimal-item"
              whileHover={{ x: 4, scale: 1.02, transition: { duration: 0.2 } }}
            >
              <div className="stat-minimal-icon">
                <TrendingUp size={32} strokeWidth={2} color="#FF8787" />
              </div>
              <div className="stat-minimal-content">
                <div className="stat-minimal-value">{successfulResults}</div>
                <div className="stat-minimal-label">Успешных</div>
              </div>
            </motion.div>
          </div>
        </motion.section>

        <ImageSlider images={[
          '/images/career-1.jpg',
          '/images/career-2.jpg',
          '/images/career-3.jpg',
          '/images/career-4.jpg'
        ]} />

        <motion.section 
          className="sessions-horizontal-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.3 }}
        >
          <div className="section-header-minimal">
            <h2 className="section-title">История консультаций</h2>
          </div>
          {sessions.length > 0 ? (
            <div className="sessions-horizontal-list">
              {sessions.map((session, index) => (
                <motion.div
                  key={session.id}
                  className="session-horizontal-item"
                  initial={{ opacity: 0, x: -20 }}
                  animate={{ opacity: 1, x: 0 }}
                  transition={{ delay: index * 0.1, duration: 0.4 }}
                  whileHover={{ 
                    x: 6, 
                    scale: 1.01, 
                    transition: { duration: 0.2 } 
                  }}
                >
                  <div className="session-horizontal-main">
                    <div className="session-horizontal-left">
                      <div className="session-horizontal-badge" style={{ background: 'linear-gradient(135deg, #FF6B6B 0%, #FFA07A 100%)' }}>
                        {session.topic}
                      </div>
                      <div className="session-horizontal-info">
                        <div className="session-horizontal-consultant">
                          <User size={18} strokeWidth={1.5} />
                          <span>{session.consultant}</span>
                        </div>
                        <div className="session-horizontal-date">
                          <Calendar size={18} strokeWidth={1.5} />
                          <span>{new Date(session.date).toLocaleDateString('ru-RU', { day: 'numeric', month: 'long' })}</span>
                        </div>
                      </div>
                    </div>
                    <div className="session-horizontal-right">
                      <div className="session-horizontal-rating">
                        {Array.from({ length: 5 }, (_, i) => (
                          <Star
                            key={i}
                            size={18}
                            fill={i < session.rating ? '#FF6B6B' : 'none'}
                            stroke={i < session.rating ? '#FF6B6B' : '#D1D5DB'}
                            strokeWidth={1.5}
                          />
                        ))}
                      </div>
                    </div>
                  </div>
                  <div className="session-horizontal-result">
                    <CheckCircle2 size={18} strokeWidth={1.5} color="#4ECDC4" />
                    <span className="result-text">{session.result}</span>
                  </div>
                  {session.review && (
                    <div className="session-horizontal-review">
                      <Star size={16} strokeWidth={1.5} fill="#FFD93D" stroke="#FFD93D" />
                      <span className="review-text">{session.review}</span>
                    </div>
                  )}
                </motion.div>
              ))}
            </div>
          ) : (
            <div className="empty-state-minimal">
              <Briefcase size={64} strokeWidth={1.5} />
              <h3 className="empty-title">Начните свой путь</h3>
              <p className="empty-text">Запишитесь на первую консультацию</p>
            </div>
          )}
        </motion.section>

        <motion.section 
          className="features-minimal-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.4 }}
        >
          <div className="features-minimal-header">
            <h2 className="section-title">Преимущества</h2>
          </div>
          <div className="features-minimal-grid">
            {[
              { icon: Target, title: 'Стратегическое планирование', text: 'Индивидуальные планы карьерного роста', accent: '#FF6B6B' },
              { icon: BarChart3, title: 'Измеримый прогресс', text: 'Отслеживание профессионального развития', accent: '#FFA07A' },
              { icon: Award, title: 'Экспертный подход', text: 'Сертифицированные карьерные консультанты', accent: '#FFB6C1' },
              { icon: Sparkles, title: 'Инновационные методики', text: 'Современные подходы к консультированию', accent: '#FF8787' }
            ].map((feature, index) => {
              const IconComponent = feature.icon
              return (
                <motion.div
                  key={feature.title}
                  className="feature-minimal-card"
                  initial={{ opacity: 0, y: 20 }}
                  animate={{ opacity: 1, y: 0 }}
                  transition={{ delay: 0.5 + index * 0.1, duration: 0.4 }}
                  whileHover={{ x: 4, scale: 1.01, transition: { duration: 0.2 } }}
                >
                  <div className="feature-minimal-icon" style={{ borderLeftColor: feature.accent }}>
                    <IconComponent size={28} strokeWidth={2} color={feature.accent} />
                  </div>
                  <div className="feature-minimal-content">
                    <h3 className="feature-minimal-title">{feature.title}</h3>
                    <p className="feature-minimal-text">{feature.text}</p>
                  </div>
                  <div className="feature-minimal-arrow">
                    <ArrowRight size={20} strokeWidth={2} color={feature.accent} />
                  </div>
                </motion.div>
              )
            })}
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default MySessions
